using System;
using System.Data;
using gov.va.med.vbecs.DAL;
using gov.va.med.vbecs.Common;
using gov.va.med.vbecs.ExceptionManagement;
using Method = System.Reflection.MethodBase;
using BLOODUNITMEDIA = gov.va.med.vbecs.Common.VbecsTables.BloodUnitMedia;
using RHFACTOR = gov.va.med.vbecs.Common.VbecsTables.RhFactor;
using BROKEN = gov.va.med.vbecs.Common.VbecsBrokenRules.FrmProcessShipmentIn;
using TABLES = gov.va.med.vbecs.Common.VbecsTables;

namespace gov.va.med.vbecs.BOL
{
	#region Header

	///<Package>Package: VBECS - VistA Blood Establishment Computer System</Package>
	///<Warning> WARNING: Per VHA Directive $VADIRECTIVE this class should not be modified</Warning>
	///<MedicalDevice> Medical Device #: $MEDDEVICENO</MedicalDevice>
	///<Developers>
	///	<Developer>John Vrooland</Developer>
	///</Developers>
	///<SiteName>Hines OIFO</SiteName>
	///<CreationDate>3/12/2003</CreationDate>
	///<Note>The Food and Drug Administration classifies this software as a medical device.  As such, it may not be changed in any way. Modifications to this software may result in an adulterated medical device under 21CFR820, the use of which is considered to be a violation of US Federal Statutes.  Acquiring and implementing this software through the Freedom of information Act requires the implementor to assume total responsibility for the software, and become a registered manufacturer of a medical device, subject to FDA regulations</Note>
	///<summary>Contains ABO/RH and UnitExpirationDate information</summary>
	///
	#endregion

	public class BloodUnitMedia : BOL.BaseBusinessObject
	{

		#region Variables

		/// <summary>
		/// bloodUnitMediaDataTable
		/// </summary>
		public DataTable bloodUnitMediaDataTable;

		/// <summary>
		/// bloodUnitMediaDataRow
		/// </summary>
		public DataRow bloodUnitMediaDataRow;

		/// <summary>
		/// _bloodUnitTests
		/// </summary>
		public DataTable dtBloodUnitTests;

		/// <summary>
		/// _aboRH
		/// </summary>
		private BOL.AboRh _aboRH;
        
		#endregion

		#region Constructors

		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/9/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5270"> 
		///		<ExpectedInput>Valid blood unit guid</ExpectedInput>
		///		<ExpectedOutput>BloodUnitMedia object initialized</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5271"> 
		///		<ExpectedInput>New guid</ExpectedInput>
		///		<ExpectedOutput>New instance of BloodUnitMedia object</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// BloodUnitMedia
		/// </summary>
		/// <param name="bloodUnitGuid"></param>
		public BloodUnitMedia(System.Guid bloodUnitGuid)
		{
			this._aboRH = new BOL.AboRh();
		
			bloodUnitMediaDataTable = DAL.BloodUnitMedia.GetBloodUnitMediaByGuid(bloodUnitGuid);

			if (bloodUnitMediaDataTable.Rows.Count > 0)
			{
				bloodUnitMediaDataRow = bloodUnitMediaDataTable.Rows[0];
				//Populate the abo rh object with the datarows values
				this.AboRh.Abo = AboRh.GetAboFromString(bloodUnitMediaDataRow[BLOODUNITMEDIA.BloodTypeCode].ToString());
				this.AboRh.RH =  AboRh.GetRHFromCode(bloodUnitMediaDataRow[BLOODUNITMEDIA.RhFactorCode].ToString());
			}
			else
			{
				MakeNew(bloodUnitGuid);
			}
			bloodUnitMediaDataRow.AcceptChanges();
			bloodUnitMediaDataTable.ColumnChanged += new DataColumnChangeEventHandler(this.OnColumnChanged);
		}

		#endregion

		#region Methods

		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/9/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5272"> 
		///		<ExpectedInput>DataRow</ExpectedInput>
		///		<ExpectedOutput>DataRow</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5273"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// LoadDataRowFromThis
		/// </summary>
		/// <param name="dtRow"></param>
		/// <returns></returns>
		public override System.Data.DataRow LoadDataRowFromThis(System.Data.DataRow dtRow)
		{
			try
			{
				if(dtRow.Table.Columns.Contains(BLOODUNITMEDIA.BloodTypeCode))
				{
					dtRow[BLOODUNITMEDIA.BloodTypeCode] = this.AboRh.Abo.ToString();
				}

				if(dtRow.Table.Columns.Contains(BLOODUNITMEDIA.BloodUnitGuid))
				{
					dtRow[BLOODUNITMEDIA.BloodUnitGuid] = this.BloodUnitGuid;
				}

				if(dtRow.Table.Columns.Contains(BLOODUNITMEDIA.BloodUnitMediaGuid))
				{
					dtRow[BLOODUNITMEDIA.BloodUnitMediaGuid] = this.BloodUnitMediaGuid;
				}

				if(dtRow.Table.Columns.Contains(BLOODUNITMEDIA.Comments))
				{
					if (this.Comments != string.Empty)
					{
						dtRow[BLOODUNITMEDIA.Comments] = this.Comments;
					}
				}

				if(dtRow.Table.Columns.Contains(BLOODUNITMEDIA.DirectedPatientGuid))
				{
					if(this.DirectedPatientGuid != Guid.Empty)
					{
						dtRow[BLOODUNITMEDIA.DirectedPatientGuid] = this.DirectedPatientGuid;
					}
				}

				if(dtRow.Table.Columns.Contains(BLOODUNITMEDIA.RhFactorCode))
				{
					dtRow[BLOODUNITMEDIA.RhFactorCode] = this.AboRh.RHFactorCode;
				}

				if(dtRow.Table.Columns.Contains(BLOODUNITMEDIA.TestingStatusId))
				{
					if (this.TestingStatusId > 0)
					{
						dtRow[BLOODUNITMEDIA.TestingStatusId] = this.TestingStatusId;
					}
				}

				if(dtRow.Table.Columns.Contains(BLOODUNITMEDIA.UnitExpirationDate))
				{
					if (this.UnitExpirationDate != System.DateTime.MinValue)
					{
						dtRow[BLOODUNITMEDIA.UnitExpirationDate] = this.UnitExpirationDate;
					}
				}

				if(dtRow.Table.Columns.Contains(BLOODUNITMEDIA.LastUpdateUser))
				{
					dtRow[BLOODUNITMEDIA.LastUpdateUser] = Common.LogonUser.LogonUserName;
				}

				if(dtRow.Table.Columns.Contains(Common.VbecsTables.BloodUnit.DivisionCode))
				{
					dtRow[Common.VbecsTables.BloodUnit.DivisionCode] = Common.LogonUser.LogonUserDivisionCode;
				}

				if(dtRow.Table.Columns.Contains(BLOODUNITMEDIA.RowVersion))
				{
					dtRow.Table.Columns[BLOODUNITMEDIA.RowVersion].ReadOnly = false;
					dtRow[BLOODUNITMEDIA.RowVersion] = this.RowVersion;
				}

				return dtRow;
			}	
			catch(Exception err)
			{
				throw new BOL.BusinessObjectException("Invalid Data", err);
			}
		}

		
		/// <summary>
		/// LoadFromDataRow
		/// </summary>
		/// <param name="dtRow"></param>
		protected override void LoadFromDataRow(System.Data.DataRow dtRow)
		{
			//Here because it needs it for that abstract class inheritance from BBO
			throw new BOL.BusinessObjectException("LoadDataRowFromThis() Not Implemented on BloodUnitMedia");		
		}

		/// <summary>
		/// MakeNew
		/// </summary>
		/// <param name="bloodUnitGuid"></param>
		private void MakeNew(System.Guid bloodUnitGuid)
		{
			if (bloodUnitMediaDataTable == null)
			{
				throw new NullReferenceException("Unit Media DataTable is Null");
			}
			bloodUnitMediaDataRow = bloodUnitMediaDataTable.NewRow();
           
			bloodUnitMediaDataRow[BLOODUNITMEDIA.BloodUnitMediaGuid] = System.Guid.NewGuid();
			bloodUnitMediaDataRow[BLOODUNITMEDIA.BloodUnitGuid] = bloodUnitGuid;
			bloodUnitMediaDataRow[BLOODUNITMEDIA.BloodTypeCode] = string.Empty;
			bloodUnitMediaDataRow[BLOODUNITMEDIA.RhFactorCode] = string.Empty;
			bloodUnitMediaDataRow[BLOODUNITMEDIA.UnitExpirationDate] = System.DateTime.MinValue;
			bloodUnitMediaDataRow[BLOODUNITMEDIA.TestingStatusId] = int.MinValue;
			bloodUnitMediaDataRow[BLOODUNITMEDIA.DirectedPatientGuid] = System.Guid.Empty;
			bloodUnitMediaDataRow[BLOODUNITMEDIA.LastUpdateDate] = System.DateTime.MinValue;
			bloodUnitMediaDataRow[BLOODUNITMEDIA.LastUpdateUser] = Common.LogonUser.LogonUserName;
			bloodUnitMediaDataTable.Rows.Add(bloodUnitMediaDataRow);
			this.IsNew = true;
		}




		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/9/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3185"> 
		///		<ExpectedInput>Valid last update function</ExpectedInput>
		///		<ExpectedOutput>Data saved</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3186"> 
		///		<ExpectedInput>Invalid blood unit media guid</ExpectedInput>
		///		<ExpectedOutput>Success indicator = false</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Save
		/// </summary>
		/// <param name="lastUpdateFunctionId">UC or calling method</param>
		public bool Save(Common.UpdateFunction lastUpdateFunctionId)
		{
			try
			{
				if (this.IsDirty || this.IsNew) 
				{

					System.Data.DataTable dtBloodUnitMedia = DAL.BloodUnitMedia.GetEmptyBloodUnitMediaTableSchema(true);
					System.Data.DataRow drBloodUnitMedia = this.LoadDataRowFromThis(dtBloodUnitMedia.NewRow());
					dtBloodUnitMedia.Rows.Add(drBloodUnitMedia);

					dtBloodUnitMedia = Common.Utility.AppendLastUpdateInformation(dtBloodUnitMedia, lastUpdateFunctionId);

					bool retval = DAL.BloodUnitMedia.Save(dtBloodUnitMedia.Rows[0]);
					if (retval) 
					{
						bloodUnitMediaDataRow.AcceptChanges();
						return true;
					}
					return false;
				}
				return true;
			}
			catch (Exception ex)
			{
				ExceptionManagement.ExceptionManager.Publish(ex);
				throw ex;
			}
		}


		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/9/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3187"> 
		///		<ExpectedInput>Valid patient guid</ExpectedInput>
		///		<ExpectedOutput>Datatable</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3188"> 
		///		<ExpectedInput>Invalid patient guid</ExpectedInput>
		///		<ExpectedOutput>Datatable with zero rows</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// GetBloodUnitsByPatient
		/// </summary>
		/// <param name="patientGuid"></param>
		/// <returns></returns>
		public static DataTable GetBloodUnitsByPatient(Guid patientGuid)
		{
			System.Data.DataTable dt = DAL.BloodUnitMedia.GetBloodUnitsByPatient(patientGuid);
			return dt;
		}

		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/9/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5278"> 
		///		<ExpectedInput>PatientGuid, int</ExpectedInput>
		///		<ExpectedOutput>DataTable</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5279"> 
		///		<ExpectedInput>invalid guid, int</ExpectedInput>
		///		<ExpectedOutput>Empty DataTable</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// GetBloodUnitsByPatient
		/// </summary>
		/// <param name="patientGuid"></param>
		/// <param name="componentClassID"></param>
		/// <returns></returns>
		public static DataTable GetBloodUnitsByPatient(Guid patientGuid, int componentClassID)
		{
			System.Data.DataTable dt = DAL.BloodUnitMedia.GetBloodUnitsByPatient(patientGuid, componentClassID);
			return dt;
		}


		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/9/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5280"> 
		///		<ExpectedInput>Valid division</ExpectedInput>
		///		<ExpectedOutput>true</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5281"> 
		///		<ExpectedInput>invalid division</ExpectedInput>
		///		<ExpectedOutput>false</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// ABORhTested - This returns a boolean value indicating if the unit has been
		/// ABO/Rh confirmation results in the current division.
		/// Implements BR_3.16, BR_73.26
		/// </summary>
		public bool IsUnitAboRHConfirmationTested(string divisionCode)
		{
			if (this.dtBloodUnitTests == null)
			{
				this.dtBloodUnitTests = BOL.BloodUnitTest.GetBloodUnitTests(this.BloodUnitGuid, divisionCode);
			}
			DataRow [] drABOInterpretations = this.dtBloodUnitTests.Select(Common.VbecsTables.BloodTestType.BloodTestTypeId + " = " + (int) Common.TestType.ABOInterp + " OR BloodTestTypeId = " + (int) Common.TestType.ABOInterpTAS);
			DataRow [] drRhInterpretations = this.dtBloodUnitTests.Select(Common.VbecsTables.BloodTestType.BloodTestTypeId + " = " + (int) Common.TestType.RhInterp + " OR BloodTestTypeId = " + (int) Common.TestType.RhInterpTAS);
			if (drABOInterpretations.Length > 0 && drRhInterpretations.Length > 0) return true;
			else return false;
		}

		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/9/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5282"> 
		///		<ExpectedInput>valid division</ExpectedInput>
		///		<ExpectedOutput>true</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5283"> 
		///		<ExpectedInput>invalid division</ExpectedInput>
		///		<ExpectedOutput>false</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// AboRHTested - This returns a boolean value indicating if the unit ABO/Rh 
		/// confirmations are consistent.
		/// Implements BR_3.16, BR_73.26
		/// </summary>
		public bool DoAboRHConfirmationsMatch(string divisionCode)
		{
			this.dtBloodUnitTests = BOL.BloodUnitTest.GetBloodUnitTests(this.BloodUnitGuid, divisionCode);

			DataRow [] drABOInterpretations = this.dtBloodUnitTests.Select("(" + Common.VbecsTables.BloodTestType.BloodTestTypeId + " = " + (int) Common.TestType.ABOInterp + " OR BloodTestTypeId = " + (int) Common.TestType.ABOInterpTAS + ") AND " + Common.VbecsTables.BloodUnitTest.RecordStatusCode + " = '" + Common.Utility.GetRecordStatusCodeCharFromEnum(Common.RecordStatusCode.Active) + "'" );
			DataRow [] drRhInterpretations = this.dtBloodUnitTests.Select("(" + Common.VbecsTables.BloodTestType.BloodTestTypeId + " = " + (int) Common.TestType.RhInterp + " OR BloodTestTypeId = " + (int) Common.TestType.RhInterpTAS + ") AND " + Common.VbecsTables.BloodUnitTest.RecordStatusCode + " = '" + Common.Utility.GetRecordStatusCodeCharFromEnum(Common.RecordStatusCode.Active) + "'"  );
	
			// Rh interpretations are only required when the unit is Rh negative.
			if (drABOInterpretations.Length == 0 || (this.AboRh.RH == Common.RH.Negative && drRhInterpretations.Length == 0))
			{
				return false;
			}
			else
			{
                //CR3558 - Pooled Rh is the same thing as RH Positive, so we create a holder here..
                Common.RH unitRH = (this.AboRh.RH == RH.PooledRH) ? RH.Positive : this.AboRh.RH;

				foreach (DataRow dr in drABOInterpretations)
				{
					if (this.AboRh.Abo != AboRh.GetAboFromString(dr[Common.VbecsTables.BloodUnitTest.TestResultId].ToString()))
					{
						return false;
					}
				}
				foreach (DataRow dr in drRhInterpretations)
				{
					// Ignore X tests -- CR3558 - check against our saved "unit RH" to account for PooledRH = Positive
					if (dr[Common.VbecsTables.BloodUnitTest.TestResultId].ToString().Trim().ToUpper() != "X" &&
						unitRH != AboRh.GetRHFromCode(dr[Common.VbecsTables.BloodUnitTest.TestResultId].ToString()))
					{
						return false;
					}
				}
			}
			return true;
		}
		/// <summary>
		/// Returns an empty table for BR_11.21
		/// </summary>
		/// <returns></returns>
		internal static DataTable GetEmptyBloodUnitMediaDataTable()
		{
			DataTable dtBloodUnitMedia = new DataTable(TABLES.BloodUnitMedia.TableName);
			dtBloodUnitMedia.Columns.Add(TABLES.BloodUnitMedia.BloodUnitMediaGuid, typeof(System.Guid));
			dtBloodUnitMedia.Columns.Add(TABLES.BloodUnitMedia.DirectedPatientGuid, typeof(System.Guid));
			dtBloodUnitMedia.Columns.Add(TABLES.BloodUnitMedia.RemoveRestrictionTechId, typeof(string));
			dtBloodUnitMedia.Columns.Add(TABLES.BloodUnitMedia.RemoveRestrictionDate, typeof(System.DateTime));
			dtBloodUnitMedia.Columns.Add(TABLES.BloodUnitMedia.Comments, typeof(string));
			dtBloodUnitMedia.Columns.Add(TABLES.OrderedUnit.DivisionCode, typeof(string));
			dtBloodUnitMedia.Columns.Add(TABLES.BloodUnitMedia.LastUpdateUser, typeof(string));
			dtBloodUnitMedia.Columns.Add(TABLES.BloodUnitMedia.RowVersion, typeof(byte[]));
			return dtBloodUnitMedia;
		}
		///<Developers>
		///	<Developer>Carrie Van Stedum</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/9/2011</CreationDate>
		/// <summary>
		/// Implements BR_11.21 to release the restriction of a unit that is volunteer allogeneic V 
		/// CR 2739
		/// </summary>
		/// <param name="bloodUnit"></param>
		/// <param name="Comment"></param>
		/// <param name="removeRestrictionDate"></param>
		/// <param name="removeRestrictionTechId"></param>
		/// <param name="updateFunctionId"></param>
		/// <returns></returns>
		///<TestCases>
		///	
		///<Case type="0" testid ="8949"> 
		///		<ExpectedInput>Valid arguments</ExpectedInput>
		///		<ExpectedOutput>boolean value indicating success/failure</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="8950"> 
		///		<ExpectedInput>InValid arguments</ExpectedInput>
		///		<ExpectedOutput>boolean value indicating success/failure</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///
		public static bool ReleaseRestriction( BloodUnit bloodUnit, string Comment, System.DateTime removeRestrictionDate, string removeRestrictionTechId, UpdateFunction updateFunctionId )
		{
			if(bloodUnit.BloodUnitMedia.DirectedPatientGuid == System.Guid.Empty) return false;
			DataTable dtBloodUnitMedia = BOL.BloodUnitMedia.GetEmptyBloodUnitMediaDataTable();

			DataRow drBloodUnitMedia = dtBloodUnitMedia.NewRow();
			drBloodUnitMedia[TABLES.BloodUnitMedia.BloodUnitMediaGuid] = bloodUnit.BloodUnitMedia.BloodUnitMediaGuid;
			drBloodUnitMedia[TABLES.BloodUnitMedia.RemoveRestrictionDate] = removeRestrictionDate;
			drBloodUnitMedia[TABLES.BloodUnitMedia.RemoveRestrictionTechId] = removeRestrictionTechId;
			drBloodUnitMedia[TABLES.OrderedUnit.DivisionCode] = Common.LogonUser.LogonUserDivisionCode;
			drBloodUnitMedia[TABLES.BloodUnitMedia.Comments] = Comment;
			drBloodUnitMedia[TABLES.BloodUnitMedia.LastUpdateUser] = Common.LogonUser.LogonUserName;
			drBloodUnitMedia[TABLES.BloodUnitMedia.RowVersion] = bloodUnit.BloodUnitMedia.RowVersion;
			dtBloodUnitMedia.Rows.Add( drBloodUnitMedia );
			
			return DAL.BloodUnitMedia.ReleaseRestriction(dtBloodUnitMedia, updateFunctionId);
			
		}
		///<Developers>
		///	<Developer>Carrie Van Stedum</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/11/2011</CreationDate>
		/// <summary>
		/// Implements the reverse of BR_11.21 to release the restriction of a unit that is volunteer allogeneic V 
		/// This is needed to satisify BR_13.03
		/// CR 3180
		/// </summary>
		/// <param name="bloodUnit"></param>
		/// <param name="updateFunctionId"></param>
		/// <returns></returns>
		/// 
		///<TestCases>
		///<Case type="0" testid ="9213"> 
		///		<ExpectedInput>Valid blood unit, valid update function id</ExpectedInput>
		///		<ExpectedOutput>Valid data table containing one row</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="9214"> 
		///		<ExpectedInput>null blood unit, unknown function id</ExpectedInput>
		///		<ExpectedOutput>data table with no rows</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///
		public static System.Data.DataTable RestoreRestriction( BloodUnit bloodUnit, UpdateFunction updateFunctionId )
		{
			//CR 3180
			DataTable dtBloodUnitMediaMirrorData = new System.Data.DataTable();
			//Get values from mirror table
			dtBloodUnitMediaMirrorData = DAL.BloodUnitMedia.GetBloodUnitMediaDirectedPatientByGuid(bloodUnit.BloodUnitGuid);
			DataTable dtBloodUnitMedia = BOL.BloodUnitMedia.GetEmptyBloodUnitMediaDataTable();
			if(dtBloodUnitMediaMirrorData != null)
			{
				if(dtBloodUnitMediaMirrorData.Rows.Count > 1)
				{
					int isFreeDirected = (System.Int32) Common.UpdateFunction.UC071FrmFreeDirectedUnit;
					System.Data.DataRow [] dtMatchedRows = dtBloodUnitMediaMirrorData.Select("LastUpdateFunctionId  = " + isFreeDirected);	
					if(dtMatchedRows.Length == 0)
					{
						//I am assuming row zero has to be from UC_11 because you cant get to UC_13 until you have put the unit in final status, this 
						//method is only called when you are removing a final status and the unit had been shipped out. Also the sproc is ordering by mirror id.
						System.Data.DataRow drBloodUnitMediaMirrorData = dtBloodUnitMediaMirrorData.Rows[0];
						DataRow drBloodUnitMedia = dtBloodUnitMedia.NewRow();
						drBloodUnitMedia[TABLES.BloodUnitMedia.BloodUnitMediaGuid] = (System.Guid) drBloodUnitMediaMirrorData[TABLES.BloodUnitMedia.BloodUnitMediaGuid];
                        //CR 3242
                        if (drBloodUnitMediaMirrorData.IsNull(TABLES.BloodUnitMedia.DirectedPatientGuid) == true)
                        {
                            drBloodUnitMedia[TABLES.BloodUnitMedia.DirectedPatientGuid] = DBNull.Value;
                        }
                        else
                        {
                            drBloodUnitMedia[TABLES.BloodUnitMedia.DirectedPatientGuid] = (System.Guid)drBloodUnitMediaMirrorData[TABLES.BloodUnitMedia.DirectedPatientGuid];
                        }
						drBloodUnitMedia[TABLES.BloodUnitMedia.RemoveRestrictionTechId] = drBloodUnitMediaMirrorData[TABLES.BloodUnitMedia.RemoveRestrictionTechId].ToString();
						drBloodUnitMedia[TABLES.OrderedUnit.DivisionCode] = Common.LogonUser.LogonUserDivisionCode;
						drBloodUnitMedia[TABLES.BloodUnitMedia.Comments] = drBloodUnitMediaMirrorData[TABLES.BloodUnitMedia.Comments].ToString();
						drBloodUnitMedia[TABLES.BloodUnitMedia.LastUpdateUser] = Common.LogonUser.LogonUserName;
						drBloodUnitMedia[TABLES.BloodUnitMedia.RowVersion] = bloodUnit.BloodUnitMedia.RowVersion;
						if(drBloodUnitMediaMirrorData.IsNull(TABLES.BloodUnitMedia.RemoveRestrictionDate) == false)
						{
							System.DateTime removeRestrictionDate = System.Convert.ToDateTime(drBloodUnitMediaMirrorData[TABLES.BloodUnitMedia.RemoveRestrictionDate].ToString());
							drBloodUnitMedia[TABLES.BloodUnitMedia.RemoveRestrictionDate] = removeRestrictionDate; 
						}
						dtBloodUnitMedia.Rows.Add( drBloodUnitMedia );
					}
					else
					{
						dtBloodUnitMedia = null;
					}
				}
			}
			return dtBloodUnitMedia;
		}

		#endregion

		#region Events

		/// <summary>
		/// TestingStatusChanged event handler
		/// </summary>
		public event EventHandler TestingStatusChanged;
		/// <summary>
		/// OnColumnChanged
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="eventArgs"></param>
		private void OnColumnChanged(object sender, DataColumnChangeEventArgs eventArgs)
		{
			this.IsDirty = true;
		}

		/// <summary>
		/// BiohazardousDonationType changed event
		/// </summary>
		protected virtual void OnTestingStatusChanged()
		{
			if(TestingStatusChanged != null)
				TestingStatusChanged(this, EventArgs.Empty);
		}
		
		#endregion

		#region Properties

		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/9/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3191"> 
		///		<ExpectedInput>Guid</ExpectedInput>
		///		<ExpectedOutput>Guid</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3192"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// BloodUnitMediaGUID
		/// </summary>
		public System.Guid BloodUnitMediaGuid
		{
			get
			{
				if (bloodUnitMediaDataRow.IsNull(BLOODUNITMEDIA.BloodUnitMediaGuid))
				{
					return System.Guid.Empty;
				}
				return (System.Guid)bloodUnitMediaDataRow[BLOODUNITMEDIA.BloodUnitMediaGuid];
			}
			set
			{
				bloodUnitMediaDataRow[BLOODUNITMEDIA.BloodUnitMediaGuid] = value;
			} 
		}

		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/9/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5286"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>BOL.AboRh</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5287"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// AboRh
		/// </summary>
		public BOL.AboRh AboRh
		{
			get
			{
				return this._aboRH;
			}
		}

		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/9/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3194"> 
		///		<ExpectedInput>Guid</ExpectedInput>
		///		<ExpectedOutput>Guid</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3199"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// BloodUnitGuid
		/// </summary>
		public System.Guid BloodUnitGuid 
		{
			get
			{
				if (bloodUnitMediaDataRow.IsNull(BLOODUNITMEDIA.BloodUnitGuid))
				{
					return System.Guid.Empty;
				}
				return (System.Guid)bloodUnitMediaDataRow[BLOODUNITMEDIA.BloodUnitGuid];
			}
			set
			{
				bloodUnitMediaDataRow[BLOODUNITMEDIA.BloodUnitGuid] = value;
			} 
		}

		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/9/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3200"> 
		///		<ExpectedInput>string</ExpectedInput>
		///		<ExpectedOutput>string</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3202"> 
		///		<ExpectedInput>invalid string</ExpectedInput>
		///		<ExpectedOutput>ArgumentException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// BloodTypeCode
		/// </summary>
		public string BloodTypeCode 
		{
			get
			{
				if (bloodUnitMediaDataRow.IsNull(BLOODUNITMEDIA.BloodTypeCode))
				{
					return String.Empty;
				}
				return bloodUnitMediaDataRow[BLOODUNITMEDIA.BloodTypeCode].ToString();
			}
			set
			{
				if (value == null || value.Length > 2)
				{
					throw new ArgumentException("Invalid BloodTypeCode argument", "value");
				}
				bloodUnitMediaDataRow[BLOODUNITMEDIA.BloodTypeCode] = value;
				this.AboRh.Abo = AboRh.GetAboFromString(value);
			} 

		}
       
		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/9/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3204"> 
		///		<ExpectedInput>System.DateTime</ExpectedInput>
		///		<ExpectedOutput>System.DateTime</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3205"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// UnitExpirationDate
		/// </summary>
		public System.DateTime UnitExpirationDate 
		{
			get
			{
				if (bloodUnitMediaDataRow.IsNull(BLOODUNITMEDIA.UnitExpirationDate))
				{
					return System.DateTime.MinValue;
				}
				return (System.DateTime)bloodUnitMediaDataRow[BLOODUNITMEDIA.UnitExpirationDate];
			}
			set
			{
				bloodUnitMediaDataRow[BLOODUNITMEDIA.UnitExpirationDate] = value;
				RuleBroken(Method.GetCurrentMethod().Name, false);
			} 
		}

		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/9/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5294"> 
		///		<ExpectedInput>shipment date, expiration date, hours, days</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5295"> 
		///		<ExpectedInput>invalid timespan</ExpectedInput>
		///		<ExpectedOutput>BrokenRule</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="0" testid ="8176"> 
		///		<ExpectedInput>shipment date, expiration date, 1 (max storage days), days</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// SetUnitExpirationDate BR_1.31
		/// </summary>
		/// <param name="shipmentDate">shipmentDate</param>
		/// <param name="expirationDate">expirationDate</param>
		/// <param name="maxStorageDays">maxStorageDays</param>
		/// <param name="maxStorageHours">maxStorageHours</param>
		/// <param name="valid">valid</param>
		public void SetUnitExpirationDate(System.DateTime shipmentDate, System.DateTime expirationDate, int maxStorageDays, int maxStorageHours, out bool valid)
		{
			System.TimeSpan timeSpan;
			if(maxStorageDays == 1 || maxStorageDays == 0)
			{
				timeSpan = expirationDate - shipmentDate;
			}
			else
			{
				System.DateTime newShipmentDate = new DateTime(shipmentDate.Year, shipmentDate.Month, shipmentDate.Day, 23, 59, 0);
				timeSpan = expirationDate - newShipmentDate;
			}
			
			System.TimeSpan maxTime = new System.TimeSpan(maxStorageDays, maxStorageHours, 0, 0);
					
			//BR_1.31
			/*A blood products expiration date shall be evaluated against the maximum storage
			 *  date/time limit for the product type and the date/time of shipment receipt 
			 * (the collection date of the product). 

			An error message is displayed when the expiration date entered would cause the unit 
			to exceed the maximum storage time if the unit had been collected on the day the 
			shipment was received: The unit expiration date post-dates the acceptable maximum
			storage time for this product. 
			The expiration date field is cleared and the user may reenter an expiration date/time
			for the product. 

			Products that have a maximum expiration date/time of less than or equal to 24 hours 
			display an expiration time 24 hours from the time of the incoming shipment. Product
			types with a maximum expiration date/time of greater than 24 hours have a default 
			expiration time of 23:59 on the last day.
			*/
			
			if(timeSpan.TotalSeconds > maxTime.TotalSeconds)
			{
				//Break rule
				RuleBroken(BROKEN.UnitExpirationDate, true);
				valid = false;
				return;
			}
			else
			{
				RuleBroken(BROKEN.UnitExpirationDate, false);
				this.UnitExpirationDate = expirationDate;
				valid = true;
				return;
			}
		}

		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/8/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7387"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="7388"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		///     Checks if the expiration date is expired and has a valid comment
		/// </summary>
		/// <param name="shipmentDate"></param>
		/// <param name="expirationDate"></param>
		public void CheckValidExpiredExpirationDate(System.DateTime shipmentDate, System.DateTime expirationDate)
		{
			System.TimeSpan timeDiff = expirationDate - shipmentDate;
			
			if(timeDiff.TotalMinutes <= -1 && this.Comments.Trim().Length == 0)
			{
				// Break rule
				RuleBroken(BROKEN.UnitExpirationDate, true);
			}
			else
			{
				// Clear rule
				RuleBroken(BROKEN.UnitExpirationDate, false);
			}
		}
		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/9/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3207"> 
		///		<ExpectedInput>int</ExpectedInput>
		///		<ExpectedOutput>int</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3208"> 
		///		<ExpectedInput>int.MinValue</ExpectedInput>
		///		<ExpectedOutput>Broken Rule is broken</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// TestingStatusId
		/// </summary>
		public int TestingStatusId 
		{
			get
			{
				if (bloodUnitMediaDataRow.IsNull(BLOODUNITMEDIA.TestingStatusId))
				{
					return 0;
				}
				return (int)bloodUnitMediaDataRow[BLOODUNITMEDIA.TestingStatusId];
			}
			set
			{
				bloodUnitMediaDataRow[BLOODUNITMEDIA.TestingStatusId] = value;
				this.OnTestingStatusChanged();
				RuleBroken(Method.GetCurrentMethod().Name, value == System.Int32.MinValue);
			} 
		}
        
		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/9/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3209"> 
		///		<ExpectedInput>Guid</ExpectedInput>
		///		<ExpectedOutput>Guid</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3210"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// DirectedPatientGuid
		/// </summary>
		public System.Guid DirectedPatientGuid
		{
			get
			{
				if (bloodUnitMediaDataRow.IsNull(BLOODUNITMEDIA.DirectedPatientGuid))
				{
					return System.Guid.Empty;
				}
				return (System.Guid)bloodUnitMediaDataRow[BLOODUNITMEDIA.DirectedPatientGuid];
			}
			set
			{
				bloodUnitMediaDataRow[BLOODUNITMEDIA.DirectedPatientGuid] = value;
			} 

		}
        

		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/9/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5300"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>string</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5301"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// LastUpdateUser
		/// </summary>
		public new System.String LastUpdateUser
		{
			get
			{
				if (bloodUnitMediaDataRow.IsNull(BLOODUNITMEDIA.LastUpdateUser)) 
				{
					return String.Empty;
				}
				return bloodUnitMediaDataRow[BLOODUNITMEDIA.LastUpdateUser].ToString();
			}
		}
        
		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/9/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5302"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>byte[]</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5303"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// RowVersion
		/// </summary>
		public new System.Byte[] RowVersion    
		{
			get
			{
				if (bloodUnitMediaDataRow.IsNull(BLOODUNITMEDIA.RowVersion)) 
				{
					return null;
				}
				return (System.Byte[])bloodUnitMediaDataRow[BLOODUNITMEDIA.RowVersion];
			}
		}   
     
		/// <summary>
		/// CR 3195
		/// </summary>
		public new System.DateTime LastUpdateDate  
		{
			get
			{
				if (bloodUnitMediaDataRow.IsNull(BLOODUNITMEDIA.LastUpdateDate)) 
				{
					return DateTime.MinValue;
				}
				return (System.DateTime)bloodUnitMediaDataRow[BLOODUNITMEDIA.LastUpdateDate];
			}
			set
			{
				bloodUnitMediaDataRow[BLOODUNITMEDIA.LastUpdateDate] = value;
			}
		}
		#endregion

	}
}
